<?php

declare(strict_types=1);

namespace App\Controller;

use Dompdf\Dompdf;
use Dompdf\Options;
use Cake\Routing\Router;
use Cake\ORM\TableRegistry;

/**
 * Consultas Controller
 *
 * @property \App\Model\Table\ConsultasTable $Consultas
 */
class ConsultasController extends AppController
{

    public function exportConsultasPdf($id = null)
    {
        $logoUrl = Router::url('/img/logoClinica.png', true);
        // Obtener los datos completos de la receta consulta
        $consulta = $this->Consultas->get($id, contain: [
            'HistoriasClinicas' => ['Pacientes'],
            'ConsultasCie' => ['Diagnosticoscie10'],
            'Doctores'
        ]);

        // Renderizar la vista HTML como contenido para el PDF
        $this->viewBuilder()->enableAutoLayout(false);
        $this->set(compact('consulta', 'logoUrl'));
        $html = $this->render('consulta_proce');

        // Configurar DomPDF para el PDF
        $options = new Options();
        $options->set('isRemoteEnabled', true);
        $dompdf = new Dompdf($options);
        $dompdf->loadHtml($html);

        // Configurar tamaño de papel
        $dompdf->setPaper('A4', 'portrait');

        // Renderizar el PDF
        $dompdf->render();

        // Descargar el archivo PDF
        $dompdf->stream("Orden_Medica_{$id}.pdf", ['Attachment' => 0]);
    }

    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $query = $this->Consultas->find();
        $query->order(['Consultas.id' => 'DESC']);
        $consultas = $this->paginate($query);

        $this->set(compact('consultas'));
    }

    /**
     * View method
     *
     * @param string|null $id Consulta id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $consulta = $this->Consultas->get($id, contain: ['DocumentosConsultas.Documentos', 'Users', 'HistoriasClinicas', 'ConsultasCie' => ['Diagnosticoscie10', 'Categoriascie10']]);
        $this->set(compact('consulta'));
        // Usar un layout diferenciado para solicitudes normales o AJAX
        if ($this->request->is('ajax')) {
            $this->viewBuilder()->setLayout('ajax');
        } else {
            $this->viewBuilder()->setLayout('default');
        }
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add($historia_id = null)
    {
        $consulta = $this->Consultas->newEmptyEntity();
        $usuario = $this->Authentication->getIdentity();
        $historia = null;

        if ($historia_id) {
            $historia = $this->Consultas->HistoriasClinicas->get($historia_id, contain: ['Pacientes']);
            $consulta->historia_id = $historia_id;
            $consulta->paciente_id = $historia->paciente_id;
        }

        if ($usuario->rol == 3) {
            $consulta->doctor_id = $usuario->doctor_id;
        }

        if ($this->request->is('post')) {
            $data = $this->request->getData();
            $data['user_id'] = $usuario->id;

            if ($usuario->rol == 3) {
                $data['doctor_id'] = $usuario->doctor_id;
            }

            if (!$historia) {
                $this->Flash->error(__('No se encontró la historia clínica.'));
                return $this->redirect($this->referer());
            }

            $data['historia_id'] = $historia->id;
            $data['paciente_id'] = $historia->paciente_id;

            // Procesar CIE
            if (!empty($data['cie_data'])) {
                $cieRelaciones = [];

                foreach ($data['cie_data'] as $json) {
                    $item = json_decode($json, true);

                    if (!empty($item['clave']) && strlen($item['clave']) > 3) {
                        // Es un CIE (clave larga)
                        $cieRelaciones[] = ['cie_id' => $item['id']];
                    } else {
                        // Es una categoría (clave corta)
                        $cieRelaciones[] = ['categoria_id' => $item['id']];
                    }
                }

                $data['consultas_cie'] = $cieRelaciones;
            }
            $consulta = $this->Consultas->patchEntity($consulta, $data, [
                'associated' => ['ConsultasCie'] // Solo asociamos CIE por ahora
            ]);
            if ($this->Consultas->save($consulta)) {

                // 📂 Manejo de documentos subidos — comprimir solo imágenes
                // Manejo de documentos subidos en add — compresión + corrección EXIF
                if (!empty($this->request->getUploadedFiles()['documentos'])) {
                    $documentos = $this->request->getUploadedFiles()['documentos'];

                    $documentosTable = $this->fetchTable('Documentos');
                    $documentosConsultasTable = $this->fetchTable('DocumentosConsultas');

                    foreach ($documentos as $archivo) {
                        if (empty($archivo->getClientFilename())) {
                            continue;
                        }

                        $dir = WWW_ROOT . "uploads/HistoriaClinica/{$consulta->historia_id}/Consultas/";
                        if (!is_dir($dir)) {
                            mkdir($dir, 0755, true);
                        }

                        $originalFilename = $archivo->getClientFilename();
                        $safeName = time() . '_' . preg_replace('/[^a-zA-Z0-9_\.-]/', '_', $originalFilename);
                        $filepath = $dir . $safeName;

                        if ($archivo->getError() !== UPLOAD_ERR_OK) {
                            $this->Flash->error(__('Error al subir el archivo: {0}', $archivo->getError()));
                            continue;
                        }

                        // Guardar stream a temp
                        $ext = strtolower(pathinfo($originalFilename, PATHINFO_EXTENSION));
                        $tmp = sys_get_temp_dir() . DIRECTORY_SEPARATOR . uniqid('upload_') . '.' . $ext;
                        file_put_contents($tmp, $archivo->getStream()->getContents());

                        // Detectar si es imagen y si GD está disponible
                        $imgInfo = @getimagesize($tmp);
                        $isImage = ($imgInfo !== false && extension_loaded('gd'));

                        if ($isImage) {
                            // Parámetros
                            $maxWidth = 1600; // ancho máximo
                            $quality = 75;    // calidad JPEG/WebP 0-100

                            // Crear resource origen
                            $type = $imgInfo[2];
                            $src = false;
                            switch ($type) {
                                case IMAGETYPE_JPEG:
                                    $src = @imagecreatefromjpeg($tmp);
                                    break;
                                case IMAGETYPE_PNG:
                                    $src = @imagecreatefrompng($tmp);
                                    break;
                                case IMAGETYPE_WEBP:
                                    if (function_exists('imagecreatefromwebp')) {
                                        $src = @imagecreatefromwebp($tmp);
                                    } else {
                                        $src = @imagecreatefromstring(file_get_contents($tmp));
                                    }
                                    break;
                                default:
                                    $src = @imagecreatefromstring(file_get_contents($tmp));
                            }

                            if ($src !== false) {
                                // --- CORRECCIÓN DE ORIENTACIÓN EXIF para JPEG ---
                                if ($type === IMAGETYPE_JPEG && function_exists('exif_read_data')) {
                                    $exif = @exif_read_data($tmp);
                                    if (!empty($exif['Orientation'])) {
                                        $orientation = (int)$exif['Orientation'];
                                        if ($orientation === 3) {
                                            $rot = imagerotate($src, 180, 0);
                                            if ($rot !== false) {
                                                imagedestroy($src);
                                                $src = $rot;
                                            }
                                        } elseif ($orientation === 6) {
                                            $rot = imagerotate($src, -90, 0);
                                            if ($rot !== false) {
                                                imagedestroy($src);
                                                $src = $rot;
                                            }
                                        } elseif ($orientation === 8) {
                                            $rot = imagerotate($src, 90, 0);
                                            if ($rot !== false) {
                                                imagedestroy($src);
                                                $src = $rot;
                                            }
                                        }
                                        // notar: orientaciones con mirror no están manejadas aquí
                                    }
                                }

                                // Obtener dimensiones desde el resource (por si rotó)
                                $width = imagesx($src);
                                $height = imagesy($src);

                                // calcular nuevo tamaño manteniendo aspecto
                                if ($width > $maxWidth) {
                                    $ratio = $height / $width;
                                    $newWidth = $maxWidth;
                                    $newHeight = (int) round($maxWidth * $ratio);
                                } else {
                                    $newWidth = $width;
                                    $newHeight = $height;
                                }

                                $dst = imagecreatetruecolor($newWidth, $newHeight);

                                // preservar transparencia PNG/WEBP
                                if ($type == IMAGETYPE_PNG || $type == IMAGETYPE_WEBP) {
                                    imagealphablending($dst, false);
                                    imagesavealpha($dst, true);
                                }

                                imagecopyresampled($dst, $src, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);

                                $saved = false;
                                if ($type == IMAGETYPE_JPEG) {
                                    $saved = imagejpeg($dst, $filepath, $quality);
                                } elseif ($type == IMAGETYPE_PNG) {
                                    $pngLevel = (int) round((100 - $quality) / 11.1111);
                                    $saved = imagepng($dst, $filepath, $pngLevel);
                                } elseif ($type == IMAGETYPE_WEBP && function_exists('imagewebp')) {
                                    $saved = imagewebp($dst, $filepath, $quality);
                                } else {
                                    $saved = imagejpeg($dst, $filepath, $quality);
                                }

                                imagedestroy($src);
                                imagedestroy($dst);

                                // fallback mover original si algo falla
                                if (!$saved) {
                                    if (!rename($tmp, $filepath)) copy($tmp, $filepath);
                                }
                            } else {
                                // no se pudo crear resource -> guardar original
                                if (!rename($tmp, $filepath)) copy($tmp, $filepath);
                            }
                        } else {
                            // No es imagen o GD no disponible -> guardar tal cual (PDF u otros)
                            if (!rename($tmp, $filepath)) copy($tmp, $filepath);
                        }

                        // limpiar temp si quedó
                        if (file_exists($tmp)) @unlink($tmp);

                        // Registrar en BD si archivo existe
                        if (file_exists($filepath)) {
                            $documento = $documentosTable->newEntity([
                                'historia_id' => $consulta->historia_id,
                                'ruta_archivo' => "uploads/HistoriaClinica/{$consulta->historia_id}/Consultas/{$safeName}",
                                'tipo' => 'consulta'
                            ]);

                            if ($documentosTable->save($documento)) {
                                $documentoConsulta = $documentosConsultasTable->newEntity([
                                    'consulta_id' => $consulta->id,
                                    'documento_id' => $documento->id
                                ]);
                                $documentosConsultasTable->save($documentoConsulta);
                            } else {
                                @unlink($filepath);
                                $this->Flash->error(__('No se pudo registrar el documento en la base de datos.'));
                            }
                        } else {
                            $this->Flash->error(__('Hubo un problema al guardar el archivo: {0}', $originalFilename));
                        }
                    }
                }

                $this->Flash->success(__('La consulta ha sido registrada con éxito.'));
                return $this->redirect(['controller' => 'Pacientes', 'action' => 'view', $historia->paciente_id, '#' => 'consultas']);
            }

            $this->Flash->error(__('No se pudo guardar la consulta. Intente de nuevo.'));
        }

        // Obtener listas de datos relacionados
        $users = $this->Consultas->Users->find('list')->all();
        $pacientes = $this->Consultas->HistoriasClinicas->find(
            'list',
            keyField: 'paciente_id',
            valueField: function ($row) {
                if ($row->paciente) {
                    return $row->paciente->nombre . ' ' . $row->paciente->apellido;
                }
                return null; // O puedes devolver "Paciente no disponible"
            }
        )->contain(['Pacientes'])
            ->distinct(['HistoriasClinicas.paciente_id'])
            ->where(['HistoriasClinicas.paciente_id IS NOT' => null])
            ->toArray();
        $doctores = $this->Consultas->Doctores->find(
            'list',
            keyField: 'id',
            valueField: 'nombre'
        )->toArray();

        $this->set(compact('consulta', 'historia_id', 'users', 'usuario', 'pacientes', 'doctores', 'historia'));
        if ($this->request->is('ajax')) {
            $this->viewBuilder()->setLayout('ajax');
        } else {
            $this->viewBuilder()->setLayout('default');
        }
    }

    /**
     * Edit method
     *
     * @param string|null $id Consulta id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $consulta = $this->Consultas->get(
            $id,
            contain: ['ConsultasCie' => ['Diagnosticoscie10', 'Categoriascie10'], 'HistoriasClinicas' => ['Pacientes'], 'DocumentosConsultas.Documentos']
        );

        $usuario = $this->Authentication->getIdentity(); // Usuario autenticado

        if ($this->request->is(['patch', 'post', 'put'])) {
            $data = $this->request->getData();

            // Eliminar todos los CIE actuales antes de agregar los nuevos
            $consultasCieTable = $this->fetchTable('ConsultasCie');
            if (is_iterable($consulta->consultas_cie)) {
                foreach ($consulta->consultas_cie as $cie) {
                    $consultasCieTable->delete($cie);
                }
            }

            // Preparar relaciones segn lo que viene del formulario
            $cieRelaciones = [];

            if (!empty($data['cie_data'])) {
                foreach ($data['cie_data'] as $json) {
                    $item = json_decode($json, true);
                    $clave = $item['clave'] ?? '';
                    $id = $item['id'];

                    if (strlen($clave) > 3) {
                        // Es un cie_id
                        $cieRelaciones[] = ['cie_id' => $id];
                    } else {
                        // Es una categoria_id
                        $cieRelaciones[] = ['categoria_id' => $id];
                    }
                }
            }
            $data['consultas_cie'] = $cieRelaciones;

            // Crear nueva entidad con los CIE frescos (sin los antiguos)
            $consulta = $this->Consultas->patchEntity($consulta, $data, [
                'associated' => ['ConsultasCie']
            ]);

            if ($this->Consultas->save($consulta)) {
                
                // Manejo de documentos subidos en edit  compresin + correccin EXIF (multifile)
                $uploadedFiles = $this->request->getUploadedFiles();
                if (!empty($uploadedFiles['documentos'])) {
                    $documentos = $uploadedFiles['documentos'];
                    
                    // Filtrar archivos vlidos
                    $documentos = array_filter($documentos, function($archivo) {
                        return !empty($archivo) && !empty($archivo->getClientFilename());
                    });

                    if (!empty($documentos)) {
                        @set_time_limit(0);
                        $documentosTable = $this->fetchTable('Documentos');
                        $documentosConsultasTable = $this->fetchTable('DocumentosConsultas');

                        foreach ($documentos as $archivo) {
                            if (empty($archivo->getClientFilename())) {
                                continue;
                            }

                            // Preparar directorio
                            $dir = WWW_ROOT . "uploads/HistoriaClinica/{$consulta->historia_id}/Consultas/";
                            if (!is_dir($dir)) {
                                mkdir($dir, 0755, true);
                            }

                            // Nombre seguro y nico para evitar colisiones
                            $originalFilename = $archivo->getClientFilename();
                            $ext = strtolower(pathinfo($originalFilename, PATHINFO_EXTENSION));
                            $base = pathinfo($originalFilename, PATHINFO_FILENAME);
                            $base = preg_replace('/[^a-zA-Z0-9_\-]/u', '_', mb_substr($base, 0, 60));
                            try {
                                $unique = time() . '_' . bin2hex(random_bytes(6));
                            } catch (\Exception $e) {
                                $unique = time() . '_' . uniqid();
                            }
                            $safeName = $unique . '_' . $base . '.' . $ext;
                            $filepath = $dir . $safeName;

                            if ($archivo->getError() !== UPLOAD_ERR_OK) {
                                $this->Flash->error(__('Error al subir el archivo: {0}', $archivo->getError()));
                                continue;
                            }

                            // Guardar stream en temp
                            $tmp = sys_get_temp_dir() . DIRECTORY_SEPARATOR . uniqid('upload_') . '.' . $ext;
                            try {
                                file_put_contents($tmp, $archivo->getStream()->getContents());
                            } catch (\Exception $e) {
                                $this->Flash->error(__('Error al guardar archivo temporal: {0}', $e->getMessage()));
                                continue;
                            }

                            // Detectar si es imagen y si GD est disponible
                            $imgInfo = @getimagesize($tmp);
                            $isImage = ($imgInfo !== false && extension_loaded('gd'));

                            if ($isImage) {
                                // Parmetros
                                $maxWidth = 1600; // ajusta si quieres menos/pesado
                                $quality = 75;    // 0-100 para JPEG/WebP

                                $type = $imgInfo[2]; // IMAGETYPE_*

                                // Crear resource desde tmp (imagecreatefrom... o fallback a imagecreatefromstring)
                                $src = false;
                                switch ($type) {
                                    case IMAGETYPE_JPEG:
                                        $src = @imagecreatefromjpeg($tmp);
                                        break;
                                    case IMAGETYPE_PNG:
                                        $src = @imagecreatefrompng($tmp);
                                        break;
                                    case IMAGETYPE_WEBP:
                                        if (function_exists('imagecreatefromwebp')) {
                                            $src = @imagecreatefromwebp($tmp);
                                        } else {
                                            $src = @imagecreatefromstring(file_get_contents($tmp));
                                        }
                                        break;
                                    default:
                                        $src = @imagecreatefromstring(file_get_contents($tmp));
                                }

                                if ($src !== false) {
                                    // --- CORRECCIN EXIF (solo JPEG): rotar PRIMERO si hay EXIF ---
                                    if ($type === IMAGETYPE_JPEG && function_exists('exif_read_data')) {
                                        $exif = @exif_read_data($tmp);
                                        if (!empty($exif['Orientation'])) {
                                            $orientation = (int)$exif['Orientation'];
                                            if ($orientation === 3) {
                                                $rot = imagerotate($src, 180, 0);
                                                if ($rot !== false) {
                                                    imagedestroy($src);
                                                    $src = $rot;
                                                }
                                            } elseif ($orientation === 6) {
                                                $rot = imagerotate($src, -90, 0);
                                                if ($rot !== false) {
                                                    imagedestroy($src);
                                                    $src = $rot;
                                                }
                                            } elseif ($orientation === 8) {
                                                $rot = imagerotate($src, 90, 0);
                                                if ($rot !== false) {
                                                    imagedestroy($src);
                                                    $src = $rot;
                                                }
                                            }
                                            // orientaciones espejo (2,4,5,7) no tratadas aqu
                                        }
                                    }

                                    // Recalcular dimensiones reales (ya rotado si aplic)
                                    $width = imagesx($src);
                                    $height = imagesy($src);

                                    // Calcular nuevo tamao manteniendo aspecto
                                    if ($width > $maxWidth) {
                                        $ratio = $height / $width;
                                        $newWidth = $maxWidth;
                                        $newHeight = (int) round($maxWidth * $ratio);
                                    } else {
                                        $newWidth = $width;
                                        $newHeight = $height;
                                    }

                                    // Crear destino preservando transparencia si aplica
                                    $dst = imagecreatetruecolor($newWidth, $newHeight);
                                    if ($type == IMAGETYPE_PNG || $type == IMAGETYPE_WEBP) {
                                        imagealphablending($dst, false);
                                        imagesavealpha($dst, true);
                                        $transparent = imagecolorallocatealpha($dst, 0, 0, 0, 127);
                                        imagefilledrectangle($dst, 0, 0, $newWidth, $newHeight, $transparent);
                                    }

                                    // Redimensionar
                                    imagecopyresampled($dst, $src, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);

                                    // Guardar segn tipo
                                    $saved = false;
                                    if ($type == IMAGETYPE_JPEG) {
                                        $saved = imagejpeg($dst, $filepath, $quality);
                                    } elseif ($type == IMAGETYPE_PNG) {
                                        $pngLevel = (int) round((100 - $quality) / 11.1111); // 0-9
                                        $saved = imagepng($dst, $filepath, $pngLevel);
                                    } elseif ($type == IMAGETYPE_WEBP && function_exists('imagewebp')) {
                                        $saved = imagewebp($dst, $filepath, $quality);
                                    } else {
                                        // fallback a JPEG
                                        $saved = imagejpeg($dst, $filepath, $quality);
                                    }

                                    imagedestroy($src);
                                    imagedestroy($dst);

                                    // fallback mover original si algo falla
                                    if (!$saved) {
                                        if (!rename($tmp, $filepath)) copy($tmp, $filepath);
                                    }
                                } else {
                                    // no se pudo crear resource -> guardar original
                                    if (!rename($tmp, $filepath)) copy($tmp, $filepath);
                                }
                            } else {
                                // No es imagen (ej. PDF) o GD no disponible -> guardar tal cual
                                if (!rename($tmp, $filepath)) copy($tmp, $filepath);
                            }

                            // borrar temp si qued
                            if (file_exists($tmp)) {
                                @unlink($tmp);
                            }

                            // Registrar en BD si archivo existe
                            if (file_exists($filepath)) {
                                $documento = $documentosTable->newEntity([
                                    'historia_id' => $consulta->historia_id,
                                    'ruta_archivo' => "uploads/HistoriaClinica/{$consulta->historia_id}/Consultas/{$safeName}",
                                    'tipo' => 'consulta'
                                ]);

                                if ($documentosTable->save($documento)) {
                                    $docConsulta = $documentosConsultasTable->newEntity([
                                        'consulta_id' => $consulta->id,
                                        'documento_id' => $documento->id
                                    ]);
                                    $documentosConsultasTable->save($docConsulta);
                                } else {
                                    @unlink($filepath);
                                    $this->Flash->error(__('No se pudo registrar el documento en la base de datos.'));
                                }
                            } else {
                                $this->Flash->error(__('Hubo un problema al guardar el archivo: {0}', $originalFilename));
                            }
                        } // foreach archivos
                    }
                }

                $this->Flash->success(__('La consulta ha sido actualizada con xito.'));
                return $this->redirect(['controller' => 'Pacientes', 'action' => 'view', $consulta->historias_clinica->paciente_id, '#' => 'consultas']);
            }

            $this->Flash->error(__('No se pudo actualizar la consulta. Intente de nuevo.'));
        }

        // Obtener listas de datos relacionados
        
        $pacientes = $this->Consultas->HistoriasClinicas->Pacientes->find(
            'list',
            keyField: 'id',
            valueField: function ($row) {
                return $row->nombre . ' ' . $row->apellido;
            }
        )->toArray();
        $doctores = $this->Consultas->Doctores->find('list', keyField: 'id', valueField: 'nombre')->toArray();

        $this->set(compact('consulta', 'pacientes', 'doctores'));
        if ($this->request->is('ajax')) {
            $this->viewBuilder()->setLayout('ajax');
        } else {
            $this->viewBuilder()->setLayout('default');
        }
    }



    /**
     * Delete method
     *
     * @param string|null $id Consulta id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $consulta = $this->Consultas->get($id);
        if ($this->Consultas->delete($consulta)) {
            $this->Flash->success(__('The consulta has been deleted.'));
        } else {
            $this->Flash->error(__('The consulta could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function buscarCie()
{
    $this->request->allowMethod(['get']);

    $query = $this->request->getQuery('q');
    $resultados = [];

    if (!empty($query)) {
        $cieDiagnosticos = $this->Consultas->Diagnosticoscie10
            ->find()
            ->select(['id', 'descripcion', 'clave'])
            ->where([
                'OR' => [
                    'LOWER(Diagnosticoscie10.descripcion) LIKE' => '%' . $query . '%',
                    'LOWER(Diagnosticoscie10.clave) LIKE' => '%' . $query . '%'
                ]
            ])
            ->limit(10)
            ->toArray();

        $cieCategorias = $this->Consultas->Categoriascie10
            ->find()
            ->select(['id', 'descripcion', 'clave'])
            ->where([
                'OR' => [
                    'LOWER(Categoriascie10.descripcion) LIKE' => '%' . $query . '%',
                    'LOWER(Categoriascie10.clave) LIKE' => '%' . $query . '%'
                ]
            ])
            ->limit(10)
            ->toArray();

        foreach ($cieCategorias as $cat) {
            $resultados[] = [
                'id' => $cat->id,
                'clave' => $cat->clave,
                'descripcion' => $cat->descripcion,
                'tipo' => 'categoria'
            ];
        }
        
        foreach ($cieDiagnosticos as $cie) {
            $resultados[] = [
                'id' => $cie->id,
                'clave' => $cie->clave,
                'descripcion' => $cie->descripcion,
                'tipo' => 'cie'
            ];
        }
    }

    return $this->response->withType('application/json')
        ->withStringBody(json_encode($resultados));
}

}
